using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;

using System.Management;

namespace JacekMatulewski.Util
{
	/// <summary>
	/// Simple bar presenting the percentage value of occupied space at choosen drive.
	/// </summary>
	/// <remarks>
	/// One can determine the drive choosing the value of <b>DriveLetter</b> property. Many drive parameters as drive type, FAT type, volume name, serial number and obviously the size of free, occupied and total size are exposed in read only properties.
	/// As an example of using <b>DiskFreeBar</b> see the application <see href="http://www.phys.uni.torun.pl/~jacek/download/dotnet.htm#diskfreenet">DiskFree .NET</see>.
	/// </remarks>
	public class DiskFreeBar : System.Windows.Forms.UserControl
	{
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		private System.Windows.Forms.ProgressBar progressBar1;

		/// <summary>
		/// Creates new instance of <b>DiskFreeBar</b>.
		/// </summary>
		public DiskFreeBar()
		{
			// This call is required by the Windows.Forms Designer.
			InitializeComponent();

			UstawPozycjePaska();
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose(bool disposing)
		{
			if (disposing)
			{
				if (components != null)
					components.Dispose();
			}
			base.Dispose(disposing);
		}

		#region Component Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.progressBar1 = new System.Windows.Forms.ProgressBar();
			this.SuspendLayout();
			// 
			// progressBar1
			// 
			this.progressBar1.Dock = System.Windows.Forms.DockStyle.Fill;
			this.progressBar1.Location = new System.Drawing.Point(0, 0);
			this.progressBar1.Name = "progressBar1";
			this.progressBar1.Size = new System.Drawing.Size(304, 72);
			this.progressBar1.TabIndex = 0;
			// 
			// DiskFreeBar
			// 
			this.Controls.Add(this.progressBar1);
			this.Name = "DiskFreeBar";
			this.Size = new System.Drawing.Size(304, 72);
			this.ResumeLayout(false);
		}
		#endregion

		/// <summary>
		/// Gets the size of free and total space aviable at drive given as the parameter.
		/// </summary>
		/// <returns>
		/// Number of bytes of free or total space on the disk pointed by <paramref name="driveLetter"></paramref>.
		/// </returns>
		/// <param name="driveLetter">
		/// The one character which characterize the drive.
		/// </param>
		/// <param name="desiredInformation">
		/// String argument passed to <see cref="System.Management.ManagementObject">System.Management.ManagementObject</see>. Possible values are "Size" or "FreeSpace".
		/// </param>
		/// <remarks>
		/// Values returned bye this method can be easier get using <see cref="DiskSize">DiskSize</see>, <see cref="DiskFreeSpace">DiskFreeSpace</see> and <see href="DiskOccupiedSpace">DiskOccupiedSpace</see> properties.
		/// </remarks>
		public static long GetDiskInfo(char driveLetter,string desiredInformation)
		{
			ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+driveLetter+":'");
			return long.Parse(mo[desiredInformation].ToString());
		}

		private char literaDysku='C';
		private long rozmiar=0;
		private long wolne=0;
		private long zajete=0;

		private void UstawPozycjePaska()
		{
			rozmiar=GetDiskInfo(literaDysku,"Size");
			wolne=GetDiskInfo(literaDysku,"FreeSpace");
			zajete=rozmiar-wolne;
			progressBar1.Value=(int)(100*zajete/rozmiar);
		}

		/// <summary>
		/// Gets or sets the letter of drive which parameters are displayed.
		/// </summary>
		/// <value>
		/// Drive symbol character.
		/// </value>
		[
			Category("Drive Information"),
			Description("Determines letter of drive which parameters are displayed.")
		]
		public char DriveLetter
		{
			set
			{
				literaDysku=char.ToUpper(value);
				UstawPozycjePaska();
			}
			get
			{
				return char.ToUpper(literaDysku);
			}
		}

		/// <summary>
		/// Refresh the component including the contents of the directory.
		/// </summary>
		public override void Refresh()
		{
		base.Refresh();
		UstawPozycjePaska();
		}

		/// <summary>
		/// Gets the disk size of drive determined by <see cref="DriveLetter">DriveLetter</see>.
		/// </summary>
		/// <value>
		/// Total size of disk in bytes.
		/// </value>
		[
			Category("Drive Information"),
			Description("The DiskSize exposes disk size of drive determined by DriveLetter.")
		]
		public long DiskSize
		{
			get
			{
				return rozmiar;
			}
		}

		/// <summary>
		/// Gets the disk free space of drive determined by <see cref="DriveLetter">DriveLetter</see>.
		/// </summary>
		/// <value>
		/// Size of free space in bytes.
		/// </value>
		[
			Category("Drive Information"),
			Description("The DiskFreeSpace exposes disk free space of drive determined by DriveLetter.")
		]
		public long DiskFreeSpace
		{
			get
			{
				return wolne;
			}
		}

		/// <summary>
		/// Gets the disk occupied space of drive determined by <see cref="DriveLetter">DriveLetter</see>.
		/// </summary>
		/// <value>
		/// Size of occupied space in bytes.
		/// </value>
		[
			Category("Drive Information"),
			Description("The DiskOccupiedSpace exposes disk occupied space of drive determined by DriveLetter.")
		]
		public long DiskOccupiedSpace
		{
			get
			{
				return zajete;
			}
		}

		/// <summary>
		/// Gets the drive volume name.
		/// </summary>
		/// <value>
		/// String containing the volume name of the drive.
		/// </value>
		[
			Category("Drive Information"),
			Description("The DriveVolumeName exposes drive volume name.")
		]
		public string DriveVolumeName
		{
			get
			{
				ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+literaDysku+":'");
				return mo["VolumeName"].ToString();
			}
		}

		/// <summary>
		/// Gets the drive type description.
		/// </summary>
		/// <value>
		/// String containing the drive type description.
		/// </value>
		[
			Category("Drive Information"),
			Description("The DriveDescription exposes the drive type description.")
		]
		public string DriveDescription
		{
			get
			{
				ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+literaDysku+":'");
				return mo["Description"].ToString();
			}
		}

		/// <summary>
		/// Gets the drive file system name.
		/// </summary>
		/// <value>
		/// String containing the drive file system code name.
		/// </value>
		[
			Category("Drive Information"),
			Description("The DriveFileSystem exposes the drive file system name.")
		]
		public string DriveFileSystem
		{
			get
			{
				ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+literaDysku+":'");
				return mo["FileSystem"].ToString();
			}
		}

		/// <summary>
		/// Gets the drive volume serial number.
		/// </summary>
		/// <value>
		/// String containing the drive volume serial number.
		/// </value>
		[
			Category("Drive Information"),
			Description("The DriveVolumeSerialNumber exposes the drive volume serial number.")
		]
		public string DriveVolumeSerialNumber
		{
			get
			{
				ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+literaDysku+":'");
				return mo["VolumeSerialNumber"].ToString();
			}
		}
	}
}
